#include <Core/Core.h>

//#undef _MULTITHREADED

#include <Job/Job.h>

//  This example demonstrates the basic aspects of Job class.
//  ---------------------------------------------------------
//  0) Instantiation with void type.
//  1) Value return semantics.
//  2) "Traditional" capture by reference method.
//  3) Worker error management.
//  4) Exception propagation.
//  5) Job cancellation/abortion mechanism. (on both multithreaded and single-threaded env.)
//  6) Constant reference access to job results in a loop.

using namespace Upp;

void PrintHelloByValue()
{
	Job<String> job;
	job.Start([=]{ return "Hello World! (by value)\n"; });
	job.Finish();
	Cout() << job.GetResult();
}

void PrintHelloByReference()
{
	String out;
	Job<void> job([&out]{ out = "Hello World! (by reference)\n";});
	job.Finish();
	Cout() << out;
}

void PrintJobError()
{
	Job<void> job;
	int i = 0;
	job & [=] { if(i == 0) throw JobError(-10, "Job Failed!"); };
	job.Finish();
	if(job.IsError()) Cerr() << "Error code = " << job.GetError() << ", Desc: " << job.GetErrorDesc() << '\n';
}

void PropagateException()
{
	Job<void> job([=]{  std::string().at(1); });
	job.Finish();
	try {
		//job.IsError();
		job.GetResult();
	}
	catch(std::exception& e) {
		Cout() << "Exception propagated by worker #" << job.GetWorkerId() << ": " << e.what() << '\n';
	}
}

#ifdef _MULTITHREADED
void CancelJob()
{
	// Multithreaded version.
	Job<void> job;
	auto work = [=] {
		while(1) {
			INTERLOCKED {Cout() << "Worker #" << GetWorkerId() << " started.\n"; }
			Sleep(Random(1000));
			if(IsJobCancelled()) {
				INTERLOCKED {
					Cout() << "Worker #" << GetWorkerId() << " received cancel signal. Cancelling job...\n";
				}
				break;
			}
			// Do some work...
		}
	};
	job.Start(pick(work));
	job.Cancel();
}
#else
void CancelJob()
{
	// Singlethreaded version. (non-blocking opereation)
	// Below example is one of the simplest ways to achive non-blocking operations with Job in a
	// single-threaded environment. A finer-grained operation would involve handling of return
	// codes. (e.g. using Job<int>)
	
	Job<void> job;
	auto work = [=] {
		if(IsJobCancelled()) {
			Cout() << "Worker #" << GetWorkerId() << " received cancel signal. Cancelling job...\n";
			throw JobError("Operation cancelled.");
		}
	};
	
	Cout() << "Worker #" << GetWorkerId() << " started. (Waiting the cancellation signal...)\n";
	const int timeout = 5000;
	auto start_time = msecs();
	while(!job.IsError()) {
		job.Start(work);
		if(msecs(start_time) >= timeout)
			job.Cancel();	// Or if you have more than one non-blocking operation in progress,
		                  	// you can call CancelJobs() global function to cancell all running jobs.
	}
}
#endif

void ComputeDivisors()
{
	Job<Vector<int>> job;
	String s;
	int number = (int) Random (500000);
	job & [=] {
		Vector<int> divisors;
		for(auto i = 1; i < number + 1; i++) {
			auto d = number % i;
			if(d == 0)
				divisors.Add(i);
		}
		return pick(divisors);
	};
	job.Finish();

	s << "Divisiors of " << number << " are: ";
	for(const auto& e : ~job)
		s << e << ' ';
	Cout() << s << '\n';
}

CONSOLE_APP_MAIN
{
	PrintHelloByValue();
	PrintHelloByReference();
	PrintJobError();
	PropagateException();
	CancelJob();
	ComputeDivisors();
}
